<?php
/**
 * Referral
 *
 * @package     AutomatorWP\Integrations\AffiliateWP\Triggers\Referral
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_AffiliateWP_Referral extends AutomatorWP_Integration_Trigger {

    public $integration = 'affiliatewp';
    public $trigger = 'affiliatewp_referral';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User gets a referral of type paid or rejected', 'automatorwp-pro' ),
            'select_option'     => __( 'User earns a referral of type <strong>paid or rejected</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Referral type. %2$s: Operation (paid or rejected). %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User gets a referral %1$s %2$s %3$s time(s)', 'automatorwp-pro' ), '{type}', '{operation}', '{times}' ),
            /* translators: %1$s: Referral type. %2$s: Operation (paid or rejected). */
            'log_label'         => sprintf( __( 'User gets a referral %1$s %2$s', 'automatorwp-pro' ), '{type}', '{operation}' ),
            'action'            => 'affwp_set_referral_status',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'type' => array(
                    'from' => 'type',
                    'fields' => array(
                        'type' => array(
                            'name' => __( 'Referral type:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options_cb' => array( $this, 'referral_types_options_cb' ),
                            'default' => 'any'
                        )
                    )
                ),
                'operation' => array(
                    'from' => 'operation',
                    'fields' => array(
                        'operation' => array(
                            'name' => __( 'Operation:', 'automatorwp-pro' ),
                            'type' => 'select',
                            'options' => array(
                                'paid'      => __( 'paid', 'automatorwp-pro' ),
                                'rejected'  => __( 'rejected', 'automatorwp-pro' ),
                            ),
                            'default' => 'paid'
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_affiliatewp_referral_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Get referral types options
     *
     * @since 1.0.0
     *
     * @return array
     */
    public function referral_types_options_cb() {

        $options = array(
            'any' => __( 'of any type', 'automatorwp-pro' )
        );

        foreach( affiliate_wp()->referrals->types_registry->get_types() as $type_id => $type ) {
            $options[$type_id] = $type['label'];
        }

        return $options;

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int    $referral_id Referral ID.
     * @param string $new_status  New referral status.
     * @param string $old_status  Old referral status.
     */
    public function listener( $referral_id, $new_status, $old_status ) {

        $referral = affwp_get_referral( $referral_id );

        // Get user id from affiliate id
        $user_id = affwp_get_affiliate_user_id( $referral->affiliate_id );

        if( $new_status === 'paid' && $old_status !== 'paid' ) {

            // Trigger the referral paid
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'referral_id'   => $referral_id,
                'type'          => $referral->type,
                'amount'        => $referral->amount,
                'operation'     => 'paid',
            ) );

        }

        if( $new_status === 'rejected' && $old_status !== 'rejected' ) {

            // Trigger the referral rejected
            automatorwp_trigger_event( array(
                'trigger'       => $this->trigger,
                'user_id'       => $user_id,
                'type'          => $referral->type,
                'amount'        => $referral->amount,
                'operation'     => 'rejected',
            ) );

        }

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if type and operation are not received
        if( ! isset( $event['type'] ) && ! isset( $event['operation'] ) ) {
            return false;
        }

        // Don't deserve if operation doesn't match with the trigger option
        if( $event['operation'] !== $trigger_options['operation'] ) {
            return false;
        }

        // Don't deserve if type doesn't match with the trigger option
        if( $trigger_options['type'] !== 'any' && $event['type'] !== $trigger_options['type'] ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 5 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['referral_id'] = ( isset( $event['referral_id'] ) ? $event['referral_id'] : 0 );

        return $log_meta;

    }

}

new AutomatorWP_AffiliateWP_Referral();